// This file is part of the AMD Anti-Lag 2 Unreal Engine Plugin.
//
// Copyright (c) 2024-2025 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include "AntiLag2Module.h"

#if PLATFORM_WINDOWS
	#include "Windows/AllowWindowsPlatformTypes.h"
	#include <Unknwnbase.h>
	#include <d3d11.h>
	#include <d3d12.h>
	#include "ffx_antilag2_dx11.h"
	#include "ffx_antilag2_dx12.h"
    #include "Windows/HideWindowsPlatformTypes.h"
#endif

#include "Misc/EngineVersionComparison.h"
#define UE_VERSION_AT_LEAST(MajorVersion, MinorVersion, PatchVersion)	\
	UE_GREATER_SORT(ENGINE_MAJOR_VERSION, MajorVersion, UE_GREATER_SORT(ENGINE_MINOR_VERSION, MinorVersion, UE_GREATER_SORT(ENGINE_PATCH_VERSION, PatchVersion, true)))

#include "Misc/CoreDelegates.h"
#if UE_VERSION_AT_LEAST(5, 7, 0)
#include "Stats/Stats.h"
#else
#include "Stats/Stats2.h"
#endif
#include "RenderingThread.h"
#include "RHI.h"
#include "RHICommandList.h"
#include "Framework/Application/SlateApplication.h"
#include "Engine/Engine.h"
#include "AntiLag2Settings.h"


// Variant of UE_VERSION_NEWER_THAN that is true if the engine version is at or later than the specified, used to better handle version differences in the codebase.


DEFINE_LOG_CATEGORY(LogAntiLag2);

DECLARE_STATS_GROUP(TEXT("AntiLag2"), STATGROUP_AntiLag2, STATCAT_Advanced);
DECLARE_CYCLE_STAT_EXTERN(TEXT("Update time"), STAT_AntiLag2UpdateTime, STATGROUP_AntiLag2, );
DEFINE_STAT(STAT_AntiLag2UpdateTime);

int32 FAntiLag2Module::RenderThreadTimeout = 0;

#if PLATFORM_WINDOWS
static void CreateDX12AntiLag2Backend(FAntiLag2Backend& Backend)
{
	Backend.Initialize = [](void) -> void*
		{
			void* Context = FMemory::Malloc(sizeof(AMD::AntiLag2DX12::Context));
			if (Context)
			{
				FMemory::Memzero(Context, sizeof(AMD::AntiLag2DX12::Context));
				if (S_OK == AMD::AntiLag2DX12::Initialize((AMD::AntiLag2DX12::Context*)Context, (ID3D12Device*)GDynamicRHI->RHIGetNativeDevice()))
				{
					UE_LOG(LogAntiLag2, Display, TEXT("AMD Anti-Lag 2 module initialized for D3D12."));
				}
				else
				{
					UE_LOG(LogAntiLag2, Error, TEXT("AMD Anti-Lag 2 module failed to initialize for D3D12."));
					FMemory::Free(Context);
					Context = nullptr;
				}
			}
			return Context;
		};
	Backend.DeInitialize = [](void* Context) -> uint64
		{
			return (uint64)AMD::AntiLag2DX12::DeInitialize((AMD::AntiLag2DX12::Context*)Context);
		};
	Backend.Update = [](void* Context, bool bEnable, unsigned int MaxFPS) -> uint32
		{
			return (uint32)AMD::AntiLag2DX12::Update((AMD::AntiLag2DX12::Context*)Context, bEnable, MaxFPS);
		};
	Backend.SetFrameGenFrameType = [](void* Context, bool bInterpolatedFrame) -> uint32
		{
			return (uint32)AMD::AntiLag2DX12::SetFrameGenFrameType((AMD::AntiLag2DX12::Context*)Context, bInterpolatedFrame);
		};
	Backend.MarkEndOfFrameRendering = [](void* Context) -> uint32
		{
			return (uint32)AMD::AntiLag2DX12::MarkEndOfFrameRendering((AMD::AntiLag2DX12::Context*)Context);
		};
}

static void CreateDX11AntiLag2Backend(FAntiLag2Backend& Backend)
{
	Backend.Initialize = [](void) -> void*
		{
			void* Context = FMemory::Malloc(sizeof(AMD::AntiLag2DX11::Context));
			if (Context)
			{
				FMemory::Memzero(Context, sizeof(AMD::AntiLag2DX11::Context));
				if (S_OK == AMD::AntiLag2DX11::Initialize((AMD::AntiLag2DX11::Context*)Context))
				{
					UE_LOG(LogAntiLag2, Display, TEXT("AMD Anti-Lag 2 module initialized for D3D11."));
				}
				else
				{
					UE_LOG(LogAntiLag2, Error, TEXT("AMD Anti-Lag 2 module failed to initialize for D3D11."));
					FMemory::Free(Context);
					Context = nullptr;
				}
			}
			return Context;
		};
	Backend.DeInitialize = [](void* Context) -> uint64
		{
			return (uint64)AMD::AntiLag2DX11::DeInitialize((AMD::AntiLag2DX11::Context*)Context);
		};
	Backend.Update = [](void* Context, bool bEnable, unsigned int MaxFPS) -> uint32
		{
			return (uint32)AMD::AntiLag2DX11::Update((AMD::AntiLag2DX11::Context*)Context, bEnable, MaxFPS);
		};
	Backend.SetFrameGenFrameType = nullptr;
	Backend.MarkEndOfFrameRendering = nullptr;
}
#endif

FAntiLag2Module::FAntiLag2Module()
: Context(nullptr)
, CachedDesiredMaxTickRate(0.f)
, bExternalCaller(false)
{
	FMemory::Memzero(Backend);
	auto CVarTimeoutForBlockOnRenderFence = IConsoleManager::Get().FindConsoleVariable(TEXT("g.TimeoutForBlockOnRenderFence"));
	FConsoleVariableDelegate EnabledChangedDelegate = FConsoleVariableDelegate::CreateStatic(&FAntiLag2Module::OnChangeAntiLag2Enabled);
	CVarFFXAntiLag2->SetOnChangedCallback(EnabledChangedDelegate);
	RenderThreadTimeout = CVarTimeoutForBlockOnRenderFence->GetInt();
}

FAntiLag2Module::~FAntiLag2Module()
{
}

void FAntiLag2Module::OnChangeAntiLag2Enabled(IConsoleVariable* Var)
{
	static auto CVarTimeoutForBlockOnRenderFence = IConsoleManager::Get().FindConsoleVariable(TEXT("g.TimeoutForBlockOnRenderFence"));
	if (CVarTimeoutForBlockOnRenderFence)
	{
		if (CVarFFXAntiLag2.GetValueOnGameThread())
		{
			RenderThreadTimeout = CVarTimeoutForBlockOnRenderFence->GetInt();
			CVarTimeoutForBlockOnRenderFence->Set(250000);
		}
		else
		{
			CVarTimeoutForBlockOnRenderFence->Set(RenderThreadTimeout);
		}
	}
}

bool FAntiLag2Module::Update()
{
	SCOPE_CYCLE_COUNTER(STAT_AntiLag2UpdateTime);
	bool bEnabled = GetEnabled();
	uint32 FrameRate = CachedDesiredMaxTickRate > 0 ? uint32(CachedDesiredMaxTickRate) : 0;
	check(Backend.Update);
	FScopeLock Lock(&Mutex);
	bEnabled = (Backend.Update(Context, bEnabled, FrameRate) == S_OK);
	return bEnabled;
}

void FAntiLag2Module::Initialize()
{
#if WITH_EDITORONLY_DATA
	if (GIsEditor)
	{
		UE_LOG(LogAntiLag2, Log, TEXT("AMD Anti-Lag 2 module not initialized in Editor."));
		return;
	}
#endif

	if (IsRHIDeviceAMD())
	{
		FString RHIName = GDynamicRHI->GetName();
#if PLATFORM_WINDOWS
		if (RHIName == TEXT("D3D12"))
		{
			CreateDX12AntiLag2Backend(Backend);
		}
		else if (RHIName == TEXT("D3D11"))
		{
			CreateDX11AntiLag2Backend(Backend);
		}
#endif

		if (Backend.Initialize && Backend.DeInitialize && Backend.Update)
		{
			Context = Backend.Initialize();
			if (Context)
			{
				if (FSlateApplication::IsInitialized())
				{
					FSlateApplication& App = FSlateApplication::Get();
					FSlateRenderer* SlateRenderer = App.GetRenderer();

#if UE_VERSION_AT_LEAST(5, 5, 0)
					BackBufferReadyToPresentDelegate = SlateRenderer->OnBackBufferReadyToPresent().AddLambda([this](class SWindow& SlateWindow, const FTextureRHIRef& BackBuffer)
#else
					BackBufferReadyToPresentDelegate = SlateRenderer->OnBackBufferReadyToPresent().AddLambda([this](class SWindow& SlateWindow, const FTexture2DRHIRef& BackBuffer)
#endif
						{
							FRHICommandListExecutor::GetImmediateCommandList().EnqueueLambda([this](FRHICommandListImmediate& cmd) mutable
								{
									if (!bExternalCaller)
									{
										if (Backend.MarkEndOfFrameRendering)
										{
											Backend.MarkEndOfFrameRendering(Context);
										}
										if (Backend.SetFrameGenFrameType)
										{
											Backend.SetFrameGenFrameType(Context, false);
										}
									}
								});
						});
				}
				else
				{
					UE_LOG(LogAntiLag2, Error, TEXT("AMD Anti-Lag 2 module failed to initialize - Slate not initialized."));

					FMemory::Free(Context);
					Context = nullptr;
				}
			}
		}
		else
		{
			UE_LOG(LogAntiLag2, Error, TEXT("AMD Anti-Lag 2 module failed to initialize: unsupported API."));
		}
	}
	else
	{
		UE_LOG(LogAntiLag2, Error, TEXT("AMD Anti-Lag 2 module failed to initialize: not an AMD GPU."));
	}
}

void FAntiLag2Module::SetEnabled(bool bInEnabled)
{
	CVarFFXAntiLag2->Set(bInEnabled);
}

bool FAntiLag2Module::GetEnabled()
{
	bool bEnabled = (CVarFFXAntiLag2.GetValueOnAnyThread() != 0) && GetAvailable();
	return bEnabled;
}

bool FAntiLag2Module::GetAvailable()
{
#if defined(WITH_FIXED_TIME_STEP_SUPPORT) && WITH_FIXED_TIME_STEP_SUPPORT != 0
	const bool bUseFixedTimeStep = FApp::IsBenchmarking() || FApp::UseFixedTimeStep();
#else
	const bool bUseFixedTimeStep = false;
#endif
	return Context != nullptr && !bUseFixedTimeStep && GEngine;
}

void FAntiLag2Module::SetFlags(uint32 Flags)
{

}

uint32 FAntiLag2Module::GetFlags()
{
	return 0;
}

bool FAntiLag2Module::HandleMaxTickRate(float DesiredMaxTickRate)
{
	bool bSkipWait = GetEnabled() && GetAvailable();
	if (GetAvailable())
	{
		CachedDesiredMaxTickRate = DesiredMaxTickRate;
		bSkipWait &= Update();
	}
	return bSkipWait;
}

void FAntiLag2Module::StartupModule()
{
	Initialize();
	IModularFeatures::Get().RegisterModularFeature(GetModularFeatureName(), this);
}

void FAntiLag2Module::MarkEndOfFrameRendering()
{
	bExternalCaller = true;
	if (Backend.MarkEndOfFrameRendering)
	{
		Backend.MarkEndOfFrameRendering(Context);
	}
}

void FAntiLag2Module::SetFrameType(bool const bInterpolatedFrame)
{
	bExternalCaller = true;
	if (Backend.SetFrameGenFrameType)
	{
		Backend.SetFrameGenFrameType(Context, bInterpolatedFrame);
	}
}

void FAntiLag2Module::ShutdownModule()
{
	if (FSlateApplication::IsInitialized())
	{
		FSlateApplication& App = FSlateApplication::Get();
		FSlateRenderer* SlateRenderer = App.GetRenderer();
		if (SlateRenderer && BackBufferReadyToPresentDelegate.IsValid())
		{
			SlateRenderer->OnBackBufferReadyToPresent().Remove(BackBufferReadyToPresentDelegate);
		}
	}

	// We own this memory, so we need to be the last ones to release - if not we will leak.
	if (Context)
	{
		uint64 RefCount = Backend.DeInitialize(Context);
		check(RefCount == 0);
		FMemory::Free(Context);
		Context = nullptr;
	}

	IModularFeatures::Get().UnregisterModularFeature(GetModularFeatureName(), this);
}

IMPLEMENT_MODULE(FAntiLag2Module, AntiLag2);
