// This file is part of the AMD Anti-Lag 2 Unreal Engine Plugin.
//
// Copyright (c) 2024-2025 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include "AntiLag2Settings.h"

#include "Misc/EngineVersionComparison.h"
// Variant of UE_VERSION_NEWER_THAN that is true if the engine version is at or later than the specified, used to better handle version differences in the codebase.
#define UE_VERSION_AT_LEAST(MajorVersion, MinorVersion, PatchVersion)	\
	UE_GREATER_SORT(ENGINE_MAJOR_VERSION, MajorVersion, UE_GREATER_SORT(ENGINE_MINOR_VERSION, MinorVersion, UE_GREATER_SORT(ENGINE_PATCH_VERSION, PatchVersion, true)))

#include "CoreMinimal.h"
#include "Interfaces/IPluginManager.h"
#include "Misc/ConfigCacheIni.h"
#if UE_VERSION_AT_LEAST(5, 1, 0)
#include "Misc/ConfigUtilities.h"
#endif

#define LOCTEXT_NAMESPACE "AntiLag2SettingsModule"

IMPLEMENT_MODULE(AntiLag2SettingsModule, AntiLag2Settings)

//------------------------------------------------------------------------------------------------------
// Console variables that control how AntiLag2 operates.
//------------------------------------------------------------------------------------------------------
TAutoConsoleVariable<int32> CVarFFXAntiLag2(
	TEXT("r.AMD.AntiLag2"),
	1,
	TEXT("Whether to enable Anti-Lag 2."),
	ECVF_RenderThreadSafe);

//-------------------------------------------------------------------------------------
// AntiLag2SettingsModule
//-------------------------------------------------------------------------------------
void AntiLag2SettingsModule::StartupModule()
{
#if UE_VERSION_AT_LEAST(5, 1, 0)
	UE::ConfigUtilities::ApplyCVarSettingsFromIni(TEXT("/Script/AntiLag2Settings.AntiLag2Settings"), *GEngineIni, ECVF_SetByProjectSetting);
#else
	ApplyCVarSettingsFromIni(TEXT("/Script/AntiLag2Settings.AntiLag2Settings"), *GEngineIni, ECVF_SetByProjectSetting);
#endif
}

void AntiLag2SettingsModule::ShutdownModule()
{
}

//-------------------------------------------------------------------------------------
// UAntiLag2Settings
//-------------------------------------------------------------------------------------
UAntiLag2Settings::UAntiLag2Settings(const FObjectInitializer& ObjectInitializer)
	: Super(ObjectInitializer)
{
}

FName UAntiLag2Settings::GetContainerName() const
{
	static const FName ContainerName("Project");
	return ContainerName;
}

FName UAntiLag2Settings::GetCategoryName() const
{
	static const FName EditorCategoryName("Plugins");
	return EditorCategoryName;
}

FName UAntiLag2Settings::GetSectionName() const
{
	static const FName EditorSectionName("AntiLag2");
	return EditorSectionName;
}

void UAntiLag2Settings::PostInitProperties()
{
	Super::PostInitProperties(); 

#if WITH_EDITOR
	if(IsTemplate())
	{
		ImportConsoleVariableValues();
	}
#endif
}

#if WITH_EDITOR


void UAntiLag2Settings::PostEditChangeProperty(struct FPropertyChangedEvent& PropertyChangedEvent)
{
	Super::PostEditChangeProperty(PropertyChangedEvent);

	if(PropertyChangedEvent.Property)
	{
		ExportValuesToConsoleVariables(PropertyChangedEvent.Property);
	}
}

#endif

#undef LOCTEXT_NAMESPACE