// This file is part of the FidelityFX Super Resolution 4.0 Unreal Engine Plugin.
//
// Copyright (c) 2023-2025 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include "FSR4MoviePipelineSettings.h"

#include "FFXFSR4TemporalUpscaling.h"

#include "MovieRenderPipelineDataTypes.h"
#include "SceneView.h"
#include "Templates/SharedPointer.h"

#define LOCTEXT_NAMESPACE "FSR4MoviePipelineSettings"

UFSR4MoviePipelineSettings::UFSR4MoviePipelineSettings()
: FSR4Quality(EFSR4MoviePipelineQuality::Quality)
{

}

void UFSR4MoviePipelineSettings::ValidateStateImpl()
{
    Super::ValidateStateImpl();

	IFFXFSR4TemporalUpscalingModule& FSR4ModuleInterface = FModuleManager::GetModuleChecked<IFFXFSR4TemporalUpscalingModule>(TEXT("FFXFSR4TemporalUpscaling"));
	if (!FSR4ModuleInterface.IsPlatformSupported(GMaxRHIShaderPlatform))
	{	
		ValidationResults.Add(FText::FromString(TEXT("FidelityFX Super Resolution 4 is unsupported on this platform.")));
		ValidationState = EMoviePipelineValidationState::Warnings;
	}
}

void UFSR4MoviePipelineSettings::GetFormatArguments(FMoviePipelineFormatArgs& InOutFormatArgs) const
{
    Super::GetFormatArguments(InOutFormatArgs);

	IFFXFSR4TemporalUpscalingModule& FSR4ModuleInterface = FModuleManager::GetModuleChecked<IFFXFSR4TemporalUpscalingModule>(TEXT("FFXFSR4TemporalUpscaling"));
	if (FSR4ModuleInterface.IsPlatformSupported(GMaxRHIShaderPlatform))
	{
		InOutFormatArgs.FileMetadata.Add(TEXT("amd/fidelityFxFSR4QualityMode"), StaticEnum<EFSR4MoviePipelineQuality>()->GetDisplayNameTextByIndex((int32)FSR4Quality).ToString());
		InOutFormatArgs.FilenameArguments.Add(TEXT("fidelityFxFSR4QualityMode"), StaticEnum<EFSR4MoviePipelineQuality>()->GetDisplayNameTextByIndex((int32)FSR4Quality).ToString());
	}
	else
	{
		InOutFormatArgs.FileMetadata.Add(TEXT("amd/fidelityFxFSR4QualityMode"), TEXT("Unsupported"));
		InOutFormatArgs.FilenameArguments.Add(TEXT("fidelityFxFSR4QualityMode"), TEXT("Unsupported"));
	}
}

void UFSR4MoviePipelineSettings::SetupViewFamily(FSceneViewFamily& ViewFamily)
{
	static IConsoleVariable* CVarScreenPercentage = IConsoleManager::Get().FindConsoleVariable(TEXT("r.ScreenPercentage"));
	static IConsoleVariable* CVarFSR4Enabled = IConsoleManager::Get().FindConsoleVariable(TEXT("r.FidelityFX.FSR4.Enabled"));
	IFFXFSR4TemporalUpscalingModule& FSR4ModuleInterface = FModuleManager::GetModuleChecked<IFFXFSR4TemporalUpscalingModule>(TEXT("FFXFSR4TemporalUpscaling"));
	if (ViewFamily.ViewMode == EViewModeIndex::VMI_Lit && CVarFSR4Enabled->GetInt())
	{
		float ScreenPercentage = FSR4Quality == EFSR4MoviePipelineQuality::Native ? 100.f : FSR4ModuleInterface.GetResolutionFraction((uint32)FSR4Quality) * 100.0f;
		if (CVarScreenPercentage)
		{
			CVarScreenPercentage->Set(ScreenPercentage, ECVF_SetByCode);
		}
	}
}
