//==============================================================================
// Copyright (c) 2017-2025 Advanced Micro Devices, Inc. All rights reserved.
/// @author AMD Developer Tools Team
/// @file
/// @brief  This file contains macro definitions to wrap existing PIX3
///         functions with AMD RGP marker support.
///
///         This requires a WinPixEventRuntime version of at least
///         1.0.200127001.
///
///         To use: Update the PIXEvents.h file to #include this header file.
///                 Then prefix the existing calls to PIXBeginEventOnContextCpu,
///                 PIXEndEventOnContextCpu and PIXSetMarkerOnContextCpu within
///                 that file to add an "Rgp" prefix (so the calls become
///                 RgpPIXBeginEventOnContextCpu, RgpPIXEndEventOnContextCpu and
///                 RgpPIXSetMarkerOnContextCpu). When using a version of
///                 WinPixEventRuntime prior to v1.0.231030001, you should also
///                 add a "Legacy" suffix to the aforementioned calls. More
///                 information, including a complete user guide, can be found
///                 in the RGP user documentation.
//==============================================================================

#ifndef _AMD_PIX3_H_
#define _AMD_PIX3_H_

#include <stdarg.h>
#include <stdio.h>
#include <tchar.h>

#include "AmdExtD3D.h"
#include "AmdExtD3DDeviceApi.h"

#define MAX_MARKER_STRING_LENGTH 1024

// Per-thread AMD extension device object using TLS.
static __declspec(thread) IAmdExtD3DDevice1* tls_pAmdExtDeviceObject = nullptr;
static __declspec(thread) bool tls_checkAmdDriver                    = true;

// This function will initialize the tls_pAmdExtDeviceObject per thread
// tls_pAmdExtDeviceObject contains the marker API.
inline void InitializeAmdExtDeviceObject(ID3D12GraphicsCommandList* pCommandList)
{
    // Return immediately if the device extension object has been created for the thread.
    if (nullptr != tls_pAmdExtDeviceObject)
    {
        return;
    }

    // Return immediately if on non-AMD system.
    if (!tls_checkAmdDriver)
    {
        return;
    }

    HMODULE hpAmdD3dDl2 = nullptr;
    BOOL    loaded      = FALSE;

    if (tls_checkAmdDriver)
    {
        loaded = GetModuleHandleEx(GET_MODULE_HANDLE_EX_FLAG_UNCHANGED_REFCOUNT, _T("amdxc64.dll"), &hpAmdD3dDl2);
    }

    if (FALSE != loaded && nullptr != hpAmdD3dDl2)
    {
        PFNAmdExtD3DCreateInterface pAmdExtD3dCreateFunc = (PFNAmdExtD3DCreateInterface)GetProcAddress(hpAmdD3dDl2, "AmdExtD3DCreateInterface");

        if (nullptr != pAmdExtD3dCreateFunc)
        {
            ID3D12Device* pDevice = nullptr;

            if (nullptr != pCommandList)
            {
                pCommandList->GetDevice(__uuidof(ID3D12Device), reinterpret_cast<void**>(&pDevice));
            }

            if (nullptr != pDevice)
            {
                // Create the extension object factory.
                IAmdExtD3DFactory* pAmdExtObject = nullptr;
                pAmdExtD3dCreateFunc(pDevice, __uuidof(IAmdExtD3DFactory), reinterpret_cast<void**>(&pAmdExtObject));

                if (nullptr != pAmdExtObject)
                {
                    // Use the extension factory object to create a device extension object that contains the marker API.
                    pAmdExtObject->CreateInterface(pDevice, __uuidof(IAmdExtD3DDevice1), reinterpret_cast<void**>(&tls_pAmdExtDeviceObject));
                    pAmdExtObject->Release();
                }
                pDevice->Release();
            }
        }
    }
    else
    {
        // Running on non-AMD hardware or missing AMD driver install.
        tls_checkAmdDriver = false;
    }
}

inline void RgpSetMarker(ID3D12GraphicsCommandList* pCommandList, PCWSTR formatString, ...)
{
    InitializeAmdExtDeviceObject(pCommandList);

    if (nullptr != tls_pAmdExtDeviceObject)
    {
        // Create a new marker string that includes all the variadic args.
        wchar_t markerString[MAX_MARKER_STRING_LENGTH];
        va_list args;
        va_start(args, formatString);
        vswprintf_s(markerString, formatString, args);
        va_end(args);

        // Convert from wchar_t to char string.
        char   markerStringInChar[MAX_MARKER_STRING_LENGTH];
        size_t retValue = 0;
        wcstombs_s(&retValue, markerStringInChar, MAX_MARKER_STRING_LENGTH, markerString, MAX_MARKER_STRING_LENGTH);

        // Set the RGP marker.
        tls_pAmdExtDeviceObject->SetMarker(pCommandList, markerStringInChar);
    }
}

inline void RgpSetMarker(ID3D12GraphicsCommandList* pCommandList, PCSTR formatString, ...)
{
    InitializeAmdExtDeviceObject(pCommandList);

    if (nullptr != tls_pAmdExtDeviceObject)
    {
        // Create a new marker string that includes all the variadic args.
        char    markerString[MAX_MARKER_STRING_LENGTH];
        va_list args;
        va_start(args, formatString);
        vsprintf_s(markerString, formatString, args);
        va_end(args);

        // Set the RGP marker.
        tls_pAmdExtDeviceObject->SetMarker(pCommandList, markerString);
    }
}

inline void RgpPushMarker(ID3D12GraphicsCommandList* pCommandList, PCWSTR formatString, ...)
{
    InitializeAmdExtDeviceObject(pCommandList);

    if (nullptr != tls_pAmdExtDeviceObject)
    {
        // Create a new marker string that includes all the variadic args.
        wchar_t markerString[MAX_MARKER_STRING_LENGTH];
        va_list args;
        va_start(args, formatString);
        vswprintf_s(markerString, formatString, args);
        va_end(args);

        // Convert from wchar_t to char string.
        char   markerStringInChar[MAX_MARKER_STRING_LENGTH];
        size_t retValue = 0;
        wcstombs_s(&retValue, markerStringInChar, MAX_MARKER_STRING_LENGTH, markerString, MAX_MARKER_STRING_LENGTH);

        // Push the RGP marker.
        tls_pAmdExtDeviceObject->PushMarker(pCommandList, markerStringInChar);
    }
}

inline void RgpPushMarker(ID3D12GraphicsCommandList* pCommandList, PCSTR formatString, ...)
{
    InitializeAmdExtDeviceObject(pCommandList);

    if (nullptr != tls_pAmdExtDeviceObject)
    {
        // Create a new marker string that includes all the variadic args.
        char    markerString[MAX_MARKER_STRING_LENGTH];
        va_list args;
        va_start(args, formatString);
        vsprintf_s(markerString, formatString, args);
        va_end(args);

        // Push the RGP marker.
        tls_pAmdExtDeviceObject->PushMarker(pCommandList, markerString);
    }
}

inline void RgpPopMarker(ID3D12GraphicsCommandList* pCommandList)
{
    InitializeAmdExtDeviceObject(pCommandList);

    if (nullptr != tls_pAmdExtDeviceObject)
    {
        tls_pAmdExtDeviceObject->PopMarker(pCommandList);
    }
}

inline void RgpSetMarker(ID3D12CommandQueue*, PCWSTR, ...)
{
    // There is no queue-based marker yet.
}

inline void RgpSetMarker(ID3D12CommandQueue*, PCSTR, ...)
{
    // There is no queue-based marker yet.
}

inline void RgpPushMarker(ID3D12CommandQueue*, PCWSTR, ...)
{
    // There is no queue-based marker yet.
}

inline void RgpPushMarker(ID3D12CommandQueue*, PCSTR, ...)
{
    // There is no queue-based marker yet.
}

inline void RgpPopMarker(ID3D12CommandQueue*)
{
    // There is no queue-based marker yet.
}

// Define three macros to wrap existing PIX3 functions when using WinPixEventRuntime v1.0.231030001 or newer.
#define RgpPIXBeginEventOnContextCpu(destination, eventSize, context, color, formatString, ...) \
    RgpPushMarker(context, formatString, args...);                                              \
    PIXBeginEventOnContextCpu(destination, eventSize, context, color, formatString, args...);

#define RgpPIXEndEventOnContextCpu(destination, context) \
    RgpPopMarker(context);                               \
    destination = PIXEndEventOnContextCpu(context);

#define RgpPIXSetMarkerOnContextCpu(destination, eventSize, context, color, formatString, ...) \
    RgpSetMarker(context, formatString, args...);                                              \
    PIXSetMarkerOnContextCpu(destination, eventSize, context, color, formatString, args...);

// Define three macros to wrap existing PIX3 functions when using a legacy version of WinPixEventRuntime (prior to v1.0.231030001).
#define RgpPIXBeginEventOnContextCpuLegacy(context, color, formatString, ...) \
    RgpPushMarker(context, formatString, args...);                            \
    PIXBeginEventOnContextCpu(context, color, formatString, args...);

#define RgpPIXEndEventOnContextCpuLegacy(context) \
    RgpPopMarker(context);                        \
    PIXEndEventOnContextCpu(context);

#define RgpPIXSetMarkerOnContextCpuLegacy(context, color, formatString, ...) \
    RgpSetMarker(context, formatString, args...);                            \
    PIXSetMarkerOnContextCpu(context, color, formatString, args...);

#endif  //_AMD_PIX3_H_
